<?php
/* --------------------------------------------------------------
   ParcelServiceApiRequestParser.php 2020-02-28
   Gambio GmbH
   http://www.gambio.de
   Copyright (c) 2020 Gambio GmbH
   Released under the GNU General Public License (Version 2)
   [http://www.gnu.org/licenses/gpl-2.0.html]
   --------------------------------------------------------------
*/

declare(strict_types=1);

namespace Gambio\Api\ParcelService;

use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceFilter;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceFilters;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSorting;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSortings;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSqlCriteria;
use Gambio\Admin\ParcelService\SqlCriteria\ParcelServiceSqlPagination;
use Psr\Http\Message\ServerRequestInterface;

/**
 * Class ParcelServiceApiRequestParser
 *
 * @package Gambio\Api\ParcelService
 */
class ParcelServiceApiRequestParser
{
    /**
     * @param ServerRequestInterface $request
     *
     * @return int
     */
    public function getPerPageFromRequest(ServerRequestInterface $request): int
    {
        return (int)$request->getQueryParam('per-page', 25);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return int
     */
    public function getPageFromRequest(ServerRequestInterface $request): int
    {
        return (int)$request->getQueryParam('page', 1);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return array
     */
    public function getFieldsFromRequest(ServerRequestInterface $request): array
    {
        $fields = $request->getQueryParam('fields');
        
        return ($fields === null) ? [] : explode(',', $fields);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return ParcelServiceSqlPagination
     */
    public function getPaginationFromRequest(ServerRequestInterface $request): ParcelServiceSqlPagination
    {
        $perPage = $this->getPerPageFromRequest($request);
        $page    = $this->getPageFromRequest($request);
        
        return ParcelServiceSqlPagination::create($perPage, ($page - 1) * $perPage);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return ParcelServiceSqlCriteria
     */
    public function getCriteriaFromRequest(ServerRequestInterface $request): ParcelServiceSqlCriteria
    {
        return ParcelServiceSqlCriteria::create($this->getFiltersFromRequest($request),
                                                $this->getSortingFromRequest($request));
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return ParcelServiceFilters
     */
    private function getFiltersFromRequest(ServerRequestInterface $request): ParcelServiceFilters
    {
        $filters = [];
        foreach ($request->getQueryParam('filter', []) as $attribute => $value) {
            $operation = (strpos($value, '*') !== false) ? 'like' : 'eq';
            if (strpos($value, '|') > 0) {
                $operation = substr($value, 0, strpos($value, '|'));
                $value     = substr($value, strpos($value, '|') + 1);
            }
            
            $filters[] = new ParcelServiceFilter($attribute, $value, $operation);
        }
        
        return ParcelServiceFilters::create(...$filters);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return ParcelServiceSortings
     */
    private function getSortingFromRequest(ServerRequestInterface $request): ParcelServiceSortings
    {
        $sortings = [];
        if ($request->getQueryParam('sort') !== null) {
            foreach (explode(',', $request->getQueryParam('sort')) as $attribute) {
                $order      = (strpos($attribute, '-') === 0) ? 'desc' : 'asc';
                $sortings[] = new ParcelServiceSorting(ltrim($attribute, '+-'), $order);
            }
        }
        
        return ParcelServiceSortings::create(...$sortings);
    }
    
    
    /**
     * @param ServerRequestInterface $request
     *
     * @return string
     */
    public function getResourceUrlFromRequest(ServerRequestInterface $request): string
    {
        return $request->getUri()->getScheme() . '://' . $request->getUri()->getHost() . $request->getUri()->getPath();
    }
}